open ExtLib
open Commons

type lab_t = string
type bin_op = Add | Sub | Mul | Div | And | Or | Shl | Shr
type larg = Var of string | Def2 of string | Untitled2
type rarg = Num of int | Def of string * int | RetCode of string * string (*function_end, callsite*)
type rexp = Bin of bin_op * larg * rarg | Not of rarg          
type inst = Arith of larg * rexp | Jmple of rarg * rarg * lab_t | PrChar of rarg | Label of lab_t
          | Jmp of lab_t  
					| While of rarg * rarg * block
					| IfLess of rarg * rarg * block * block 
					| BeginScope of string option * bool | EndScope
					| Fun of string * string list * block
					| DenseFun of string * string list * block
					| Prepare of string * string (*function, callsite*)
					| FastCall of string * string (*function, callsite*)
					| Call of string * rexp list
					| Print of string
					| PrintOnce of string
           
and block = inst list;;

let binop_s = function
	Add -> "+" | Sub -> "-" | Mul->"*" | Div->"/" | And->"&" | Or->"|" | Shl->"<<" | Shr->">>"

let larg_s = function
	| Var s -> "@" ^ s | Def2 s -> Printf.sprintf "(%s:=2)" s | Untitled2 -> "(?:=2)"

let larg1_s = function
	| Var s -> "@" ^ s | Def2 s -> Printf.sprintf "(%s:=1)" s | Untitled2 -> "(?:=1)"

let rarg_s = function
	| Num n -> string_of_int n | Def(s,n) -> Printf.sprintf "(%s:=%d)" s n
	| RetCode(fend, cs) -> Printf.sprintf "return from %s to %s" fend cs

let rexp_s = function
	| Bin(op, l, r) -> Printf.sprintf "%s %s %s" (larg_s l) (binop_s op) (rarg_s r)
	| Not r -> "~ " ^ (rarg_s r)

let rec inst_s = function
	| Arith(l, e) -> Printf.sprintf "%s = %s" (larg1_s l) (rexp_s e)
	| Jmple(a1,a2,lab) -> Printf.sprintf "if %s < %s jump to %s" (rarg_s a1) (rarg_s a2) lab
	| PrChar a -> Printf.sprintf "prchar %s" (rarg_s a)
	| Label s -> Printf.sprintf "Label %s:" s
	| Jmp s -> Printf.sprintf "jmp %s" s
	| While(a1,a2,code) -> Printf.sprintf "while %s < %s\n[%s\n]" (rarg_s a1) (rarg_s a2) (code_s code)
	| IfLess(a1,a2,tcode,ecode) -> Printf.sprintf "if %s < %s then\n[%s\n] else [%s\n]" 
																	(rarg_s a1) (rarg_s a2) (code_s tcode) (code_s ecode)
  | BeginScope(Some s, _) -> "fun " ^ s ^ " {"
  | BeginScope(None, _) -> "{"
	| EndScope -> "}"		
	| Fun(nm, params, code) -> Printf.sprintf "fun %s(%s) {\n%s\n}" nm (String.join ", " params) (code_s code)
	| DenseFun(nm, params, code) -> Printf.sprintf "dfun %s(%s) {\n%s\n}" nm (String.join ", " params) (code_s code)
	| Prepare(fn, cs) -> Printf.sprintf "prepare calling %s from %s" fn cs
	| FastCall(fn, cs) -> Printf.sprintf "call %s from %s" fn cs
	| Call(fn, rlist) -> Printf.sprintf "call %s(%s)" fn (List.map rexp_s rlist |> String.join ", ")
	| Print s -> Printf.sprintf "print '%s'" s 															
	| PrintOnce s -> Printf.sprintf "print_once '%s'" s 															

and code_s code = List.map inst_s code |> String.join "\n";;
